document.addEventListener("DOMContentLoaded", () => {
  const sidebar = document.getElementById("sidebar");
  const container = document.getElementById("menuItems");

  /* Capture original DOM order (restore to this on hover) */
  const origOrder = Array.from(container.children);

  /* Mutable working list */
  let items = [...origOrder];

  /* Shuffle control */
  let shuffleTimer = null;
  const SHUFFLE_INTERVAL = 3000; // ms

  /* State */
  let isHovered = false;
  let isAnimating = false;

  /* Track current animation objects so we can cancel on hover */
  let currentAnimations = []; // array of Animation objects
  let animPairA = null;
  let animPairB = null;
  let swapCommitted = false;

  /* ------------- Helpers ------------- */

  function clearShuffleTimer() {
    if (shuffleTimer) {
      clearInterval(shuffleTimer);
      shuffleTimer = null;
    }
  }

  function startShuffleTimer() {
    clearShuffleTimer();
    shuffleTimer = setInterval(runShuffleCycle, SHUFFLE_INTERVAL);
  }

  function cancelCurrentAnimations() {
    currentAnimations.forEach((anim) => {
      try {
        anim.cancel();
      } catch (_) {}
    });
    currentAnimations = [];
  }

  function restoreOriginalOrder() {
    origOrder.forEach((node) => container.appendChild(node));
    items = Array.from(container.children);
  }

  function commitCurrentSwapIfNeeded() {
    if (isAnimating && animPairA && animPairB && !swapCommitted) {
      swapElements(animPairA, animPairB);
      swapCommitted = true;
      items = Array.from(container.children);
    }
  }

  /* Swap two children in the list */
  function swapElements(el1, el2) {
    const el1Next = el1.nextSibling;
    const el2Next = el2.nextSibling;

    if (el1Next === el2) {
      // adjacent el1 before el2
      container.insertBefore(el2, el1);
      return;
    }
    if (el2Next === el1) {
      // adjacent el2 before el1
      container.insertBefore(el1, el2);
      return;
    }
    const placeholder = document.createElement("div");
    container.insertBefore(placeholder, el1);
    container.insertBefore(el1, el2Next);
    container.insertBefore(el2, placeholder);
    container.removeChild(placeholder);
  }

  /* Get random upper/lower pair */
  function getRandomPair() {
    if (items.length < 2) return null;
    const shuffled = [...items].sort(() => Math.random() - 0.5);
    const a = shuffled[0];
    const b = shuffled[1];
    const ia = items.indexOf(a);
    const ib = items.indexOf(b);
    return ia < ib ? [a, b] : [b, a]; // ensure [upper, lower]
  }

  /* ----- Core WAAPI swap animation ----- */
  async function animateSwapWAAPI(itemUpper, itemLower) {
    isAnimating = true;
    animPairA = itemUpper;
    animPairB = itemLower;
    swapCommitted = false;

    // Measure
    const rU = itemUpper.getBoundingClientRect();
    const rL = itemLower.getBoundingClientRect();
    const verticalOffset = rL.top - rU.top; // positive distance downward
    const horizontalOffset = sidebar.offsetWidth + 20;

    // Durations (ms)
    const d1 = 400; // out
    const d2 = 400; // vertical cross outside
    const d3 = 400; // back in

    // Step 1: slide out horizontally
    const animU1 = itemUpper.animate(
      [
        { transform: "translate(0,0)" },
        { transform: `translate(${-horizontalOffset}px,0)` },
      ],
      { duration: d1, easing: "ease" }
    );

    const animL1 = itemLower.animate(
      [
        { transform: "translate(0,0)" },
        { transform: `translate(${horizontalOffset}px,0)` },
      ],
      { duration: d1, easing: "ease" }
    );

    currentAnimations.push(animU1, animL1);
    await Promise.all([animU1.finished, animL1.finished]).catch(() => {});

    if (isHovered) {
      finishEarly();
      return;
    }

    // Step 2: move vertically while outside
    const animU2 = itemUpper.animate(
      [
        { transform: `translate(${-horizontalOffset}px,0)` },
        { transform: `translate(${-horizontalOffset}px,${verticalOffset}px)` },
      ],
      { duration: d2, easing: "ease" }
    );

    const animL2 = itemLower.animate(
      [
        { transform: `translate(${horizontalOffset}px,0)` },
        { transform: `translate(${horizontalOffset}px,${-verticalOffset}px)` },
      ],
      { duration: d2, easing: "ease" }
    );

    currentAnimations.push(animU2, animL2);
    await Promise.all([animU2.finished, animL2.finished]).catch(() => {});

    if (isHovered) {
      finishEarly();
      return;
    }

    // Step 3: slide back in swapped vertically
    const animU3 = itemUpper.animate(
      [
        { transform: `translate(${-horizontalOffset}px,${verticalOffset}px)` },
        { transform: `translate(0,${verticalOffset}px)` },
      ],
      { duration: d3, easing: "ease" }
    );

    const animL3 = itemLower.animate(
      [
        { transform: `translate(${horizontalOffset}px,${-verticalOffset}px)` },
        { transform: `translate(0,${-verticalOffset}px)` },
      ],
      { duration: d3, easing: "ease" }
    );

    currentAnimations.push(animU3, animL3);
    await Promise.all([animU3.finished, animL3.finished]).catch(() => {});

    if (isHovered) {
      finishEarly();
      return;
    }

    // Commit swap + cleanup
    finalizeSwap(itemUpper, itemLower, verticalOffset);
  }

  /* Called if hover interrupts mid-animation */
  function finishEarly() {
    cancelCurrentAnimations();
    commitCurrentSwapIfNeeded(); // commit if mid-flight (so DOM valid)
    resetTransformsAll(); // snap to layout
    isAnimating = false;
    animPairA = animPairB = null;
  }

  /* Called when animation completes normally */
  function finalizeSwap(itemUpper, itemLower) {
    cancelCurrentAnimations();
    if (!swapCommitted) {
      swapElements(itemUpper, itemLower);
      swapCommitted = true;
    }
    items = Array.from(container.children);
    resetTransformsAll(); // remove any residual transforms
    isAnimating = false;
    animPairA = animPairB = null;
  }

  /* Clear transforms on every item */
  function resetTransformsAll() {
    items.forEach((el) => {
      el.style.transform = "none"; // keep as 'none' (no transition here)
    });
    // force reflow so next animation starts from clean 0,0
    container.offsetWidth;
  }

  /* ----- Shuffle cycle ----- */
  function runShuffleCycle() {
    if (isHovered || isAnimating) return;
    const pair = getRandomPair();
    if (!pair) return;
    const [upper, lower] = pair;
    animateSwapWAAPI(upper, lower);
  }

  /* ----- Hover behaviour ----- */
  sidebar.addEventListener("mouseenter", () => {
    isHovered = true;
    clearShuffleTimer();
    // stop animations
    finishEarly();
    // always restore to original HTML order per your spec
    restoreOriginalOrder();
    resetTransformsAll();
  });

  sidebar.addEventListener("mouseleave", () => {
    isHovered = false;
    // give layout a moment to settle before restarting
    setTimeout(() => {
      if (!isHovered) startShuffleTimer();
    }, 150);
  });

  /* Start! */
  startShuffleTimer();
});
